<?php

/**
 * Sibs Payment Gateway.
 */

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use Sibs2\sibsConfig;
use Sibs2\sibsConstants;
use Sibs2\sibsFunctions;

defined('ABSPATH') || exit;

/**
 * Sibs payment method integration.
 *
 * @since 1.5.0
 */
final class WC_Sibs_Blocks_Support extends AbstractPaymentMethodType
{
    /**
     * Name of the payment method.
     *
     * @var string
     */
    // protected $name = 'sibs_stg_cc';
    protected $name;

    public function __construct($method_id)
    {
        $this->name = $method_id;
    }

    /**
     * Initializes the payment method type.
     */
    public function initialize()
    {
        $this->settings = sibsConfig::getPaymentOption($this->get_name());
    }

    /**
     * Returns payment method.
     *
     * @return mixed
     */
    public function get_payment_method()
    {
        $payment_gateways_class = WC()->payment_gateways();
        $payment_gateways       = $payment_gateways_class->payment_gateways();

        return $payment_gateways[$this->get_name()] ?? null;
    }

    /**
     * Returns if this payment method should be active. If false, the scripts will not be enqueued.
     *
     * @return bool
     */
    public function is_active()
    {
        $payment_method = $this->get_payment_method();

        return $payment_method && method_exists($payment_method, 'is_available')
            ? $payment_method->is_available()
            : false;
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method.
     *
     * @return array
     */
    public function get_payment_method_script_handles()
    {
        $asset_path   = constant('WC_GATEWAY_SIBS_PATH') . '/build/' . $this->get_name() . '.asset.php';
        $version      = constant('SIBS_VERSION');
        $dependencies = [];

        if (file_exists($asset_path)) {
            $asset        = require $asset_path;
            $version      = is_array($asset) && isset($asset['version'])
                ? $asset['version']
                : $version;
            $dependencies = is_array($asset) && isset($asset['dependencies'])
                ? $asset['dependencies']
                : $dependencies;
        }

        wp_register_script(
            'wc-' . $this->get_name() . '-blocks-integration',
            constant('WC_GATEWAY_SIBS_URL') . '/build/' . $this->get_name() . '.js',
            $dependencies,
            $version,
            true
        );
        wp_set_script_translations(
            'wc-' . $this->get_name() . '-blocks-integration',
            'woocommerce-gateway-sibs'
        );

        return ['wc-' . $this->get_name() . '-blocks-integration'];
    }

    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script.
     *
     * @return array
     */
    public function get_payment_method_data()
    {
        $payment_method   = $this->get_payment_method();
        $logo_path        = constant('WC_GATEWAY_SIBS_URL') . '/assets/images/';
        $logo_name        = str_replace('sibs_stg_', '', strtolower($this->get_name()));
        $logos            = [];

        if (isset($payment_method->settings['card_types'])) {
            foreach ($payment_method->settings['card_types'] as $logo) {
                $multi_logo_name = str_replace('sibs_stg_', '', strtolower($logo));
                $logos[] = $logo_path . $multi_logo_name . '.svg';
            }
        } else {
            $logos[] = $logo_path . $logo_name . '.svg';
        }

        $oneclick        = sibsConfig::getPaymentOption(sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD, 'tokenization');
        $checkoutLabel   = '';
        $lang            = sibsFunctions::get_language();
        $pdf             = "sibs_terms_and_conditions_{$lang}.pdf";
        $siteUrl         = constant('SIBS_PLUGIN_DIR');
        $pdfUrl          = "{$siteUrl}assets/files/$pdf";

        if ($oneclick === sibsConstants::SIBS_TOKEN_ONECLICK && get_current_user_id()) {
            $termsAndConditionsLinkLabel  = __('Terms and conditions', sibsConstants::SIBS_TRANSLATE);
            $oneclickExplanationLinkLabel = __('OneClick', sibsConstants::SIBS_TRANSLATE);
            $oneclickExplanationLink      = "<a href='{$pdfUrl}' target='_blank'>{$oneclickExplanationLinkLabel}</a>";
            $checkoutLabel                = sprintf(
                __('I agree to the %s and will adhere to them unconditionally for %s payment.', sibsConstants::SIBS_TRANSLATE),
                $termsAndConditionsLinkLabel,
                $oneclickExplanationLink,
            );
        }

        return [
            'title'               => $payment_method->title,
            'description'         => $payment_method->settings['payment_desc'] ?? '',
            'supports'            => $payment_method->supports,
            'checkoutLabel'       => $checkoutLabel,
            'logos'               => $logos,
        ];
    }
}
