<?php

namespace Sibs2;

defined('ABSPATH') || exit;

class sibsPaymentSettings
{
    /**
     * SIBS Create Backend Payment Settings.
     *
     * @param string $payment_id
     * @return array
     */
    public static function sibs_create_backend_payment_settings($payment_id)
    {
        $form_fields = [];

        $form_fields['enabled'] =  [
            'title'      => __('Active', sibsConstants::SIBS_TRANSLATE),
            'type'       => 'checkbox',
            'default'    => '',
            'desc'       => '',
        ];

        $form_fields = array_merge_recursive(
            $form_fields,
            self::add_card_fields_in_form($payment_id),
            self::add_xpay_fields_in_form($payment_id),
            self::add_auth_db_fields_in_form($payment_id),
            self::add_pay_by_link_fields_in_form($payment_id),
            self::add_blik_fields_in_form($payment_id),
            self::add_tokenization_fields_in_form($payment_id),
            self::add_recurring_fields_in_form($payment_id),
            self::add_recurring_credit_card_fields_in_form($payment_id)
        );

        $form_fields['payment_desc'] = [
            'title'    => __('Payment Description', sibsConstants::SIBS_TRANSLATE),
            'type'     => 'textarea',
            'default'  => '',
            'desc'     => '',
        ];

        return $form_fields;
    }

    /**
     * Add Xpay Fields in Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_xpay_fields_in_form($payment_id)
    {
        $form_fields        = [];
        $cardPaymentMethods = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_XPAY,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_GPAY,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_APAY,
        ];

        if (in_array($payment_id, $cardPaymentMethods)) {
            $form_fields['card_types'] = [
                'title'    => __('Card Logos to present on Checkout', sibsConstants::SIBS_TRANSLATE),
                'type'     => 'multiselect',
                'css'      => 'height:12em;',
                'options'  => [
                    sibsConstants::SIBS_PAYMENT_METHOD_GPAY => __('Google Pay', sibsConstants::SIBS_TRANSLATE),
                    sibsConstants::SIBS_PAYMENT_METHOD_APAY => __('Apple Pay', sibsConstants::SIBS_TRANSLATE),
                ],
                'default'  => [sibsConstants::SIBS_PAYMENT_METHOD_GPAY, sibsConstants::SIBS_PAYMENT_METHOD_APAY],
                'desc'     => esc_attr(__('Hold Ctrl while selecting the card you want to activate the payment gateway for the selected card', sibsConstants::SIBS_TRANSLATE)),
            ];
        }

        return $form_fields;
    }

    /**
     * Add Card Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_card_fields_in_form($payment_id)
    {
        $form_fields        = [];
        $cardPaymentMethods = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD_SAVED,
        ];

        if (in_array($payment_id, $cardPaymentMethods)) {
            $form_fields['card_types'] = [
                'title'    => __('Card Logos to present on Checkout', sibsConstants::SIBS_TRANSLATE),
                'type'     => 'multiselect',
                'css'      => 'height:12em;',
                'options'  => [
                    //'AMEX'          => __('American Express', sibsConstants::SIBS_TRANSLATE),
                    //'MAESTRO'       => __('Maestro', sibsConstants::SIBS_TRANSLATE),
                    'MASTER'        => __('MasterCard', sibsConstants::SIBS_TRANSLATE),
                    // 'MASTERDEBIT'   => __('MasterCard Debit', sibsConstants::SIBS_TRANSLATE),
                    'VISA'          => __('VISA', sibsConstants::SIBS_TRANSLATE),
                    // 'VISADEBIT'     => __('VISA Debit', sibsConstants::SIBS_TRANSLATE),
                    //'VISAELECTRON'  => __('VISA Electron', sibsConstants::SIBS_TRANSLATE),
                    //'VPAY'          => __('V PAY', sibsConstants::SIBS_TRANSLATE),
                ],
                'default'  => ['MASTER', 'VISA'],
                'desc'     => esc_attr(__('Hold Ctrl while selecting the card you want to activate the payment gateway for the selected card', sibsConstants::SIBS_TRANSLATE)),
            ];
        }

        return $form_fields;
    }

    /**
     * Add Auth Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_auth_db_fields_in_form($payment_id)
    {
        $form_fields = [];

        $payment_auth_db = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD_SAVED,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_DIRECT_DEBIT,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_DIRECT_DEBIT_SAVED,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD,
            // sibsConstants::SIBS_PAYMENT_METHOD_ORDER_XPAY,
            // sibsConstants::SIBS_PAYMENT_METHOD_ORDER_APAY,
            // sibsConstants::SIBS_PAYMENT_METHOD_ORDER_GPAY,
        ];

        if (in_array($payment_id, $payment_auth_db)) {
            $form_fields['trans_mode'] = [
                'title'   => __('Transaction-Mode', sibsConstants::SIBS_TRANSLATE),
                'type'    => 'select',
                'options' => [
                    sibsConstants::SPG_PAYMENT_TYPE_PURS  => __('Debit', sibsConstants::SIBS_TRANSLATE),
                    sibsConstants::SPG_PAYMENT_TYPE_AUTH  => __('Pre-Authorization', sibsConstants::SIBS_TRANSLATE),
                ],
                'default'  => sibsConstants::SPG_PAYMENT_TYPE_PURS,
                'desc'     => '',
            ];
        }

        return $form_fields;
    }

    /**
     * Add Pay By Link Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_pay_by_link_fields_in_form($payment_id)
    {
        $form_fields = [];

        // $payment_pay_by_link = [
        //     sibsConstants::SIBS_PAYMENT_METHOD_ORDER_PAY_BY_LINK,
        //     sibsConstants::SIBS_PAYMENT_METHOD_ORDER_PAY_BY_LINK_KEVIN,
        // ];

        // if (in_array($payment_id, $payment_pay_by_link)) {
        //     $form_fields['sibs_payment_date_value'] = [
        //         'title'    => __('SIBS Pbl Expiration Value', sibsConstants::SIBS_TRANSLATE),
        //         'type'     => 'text',
        //         'default'  => '60',
        //         'desc'     => '',
        //     ];

        //     $form_fields['sibs_payment_date_unit'] = [
        //         'title'   => __('SIBS Pbl Expiration Unit', sibsConstants::SIBS_TRANSLATE),
        //         'type'    => 'select',
        //         'options' => [
        //             'minute'    => __('Minute', sibsConstants::SIBS_TRANSLATE),
        //             'hour'      => __('Hour', sibsConstants::SIBS_TRANSLATE),
        //             'day'       => __('Day', sibsConstants::SIBS_TRANSLATE),
        //             'month'     => __('Month', sibsConstants::SIBS_TRANSLATE),
        //         ],
        //         'default'  => 'minute',
        //         'desc'     => '',
        //     ];
        // }

        return $form_fields;
    }

    /**
     * Add Blik Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_blik_fields_in_form($payment_id)
    {
        $form_fields = [];

        // $payment_blik = [
        //     sibsConstants::SIBS_PAYMENT_METHOD_ORDER_BLIK,
        // ];

        // if (in_array($payment_id, $payment_blik)) {
        //     $form_fields['sibs_number_of_retries'] = [
        //         'title'    => __('SIBS Blik Number of Retries', sibsConstants::SIBS_TRANSLATE),
        //         'type'     => 'text',
        //         'default'  => '30',
        //         'desc'     => '',
        //     ];

        //     $form_fields['sibs_retry_delay'] = [
        //         'title'    => __('SIBS Blik Retry Delay (seconds)', sibsConstants::SIBS_TRANSLATE),
        //         'type'     => 'text',
        //         'default'  => '1',
        //         'desc'     => '',
        //     ];
        // }

        return $form_fields;
    }

    /**
     * Add Tokenization Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_tokenization_fields_in_form($payment_id)
    {
        $form_fields = [];

        $payment_tokenization = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD_SAVED,
        ];

        if (in_array($payment_id, $payment_tokenization)) {
            $form_fields['tokenization'] = [
                'title'   => __('Tokenization', sibsConstants::SIBS_TRANSLATE),
                'type'    => 'select',
                'options' => [
                    'none'              => __('None', sibsConstants::SIBS_TRANSLATE),
                    'tokenization'      => __('Tokenize', sibsConstants::SIBS_TRANSLATE),
                    'oneclick'          => __('OneClick', sibsConstants::SIBS_TRANSLATE),
                ],
                'default' => 'none',
            ];
        }

        return $form_fields;
    }

    /**
     * Add Recurring Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_recurring_fields_in_form($payment_id)
    {
        $form_fields = [];

        $payment_recurring = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD_SAVED,
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_DIRECT_DEBIT_SAVED,
        ];

        if (in_array($payment_id, $payment_recurring)) {
            $form_fields['amount_registration'] = [
                'title'       => __('Amount for Registration', sibsConstants::SIBS_TRANSLATE),
                'type'        => 'text',
                'default'     => '',
                'desc'        => __('Amount that is debited and refunded when a shopper registers a payment method without purchase', sibsConstants::SIBS_TRANSLATE),
            ];
        }

        return $form_fields;
    }

    /**
     * Add Recurring Fields In Form.
     *
     * @param string $payment_id
     * @return array
     */
    public static function add_recurring_credit_card_fields_in_form($payment_id)
    {
        $form_fields = [];

        $payment_recurring_credit_card = [
            sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD_SAVED,
        ];

        if (in_array($payment_id, $payment_recurring_credit_card)) {
            $form_fields['multichannel'] = [
                'title'   => __('Multichannel', sibsConstants::SIBS_TRANSLATE),
                'type'    => 'select',
                'options' => [
                    'true'  => __('Yes', sibsConstants::SIBS_TRANSLATE),
                    'false' => __('No', sibsConstants::SIBS_TRANSLATE),
                ],
                'default'  => 'false',
                'desc'     => __('If activated, repeated recurring payments are handled by the alternative channel', sibsConstants::SIBS_TRANSLATE),
            ];

            $form_fields['channel_moto'] = [
                'title'    => __('Entity-ID MOTO', sibsConstants::SIBS_TRANSLATE),
                'type'     => 'text',
                'default'  => '',
                'desc'     => __('Alternative channel for recurring payments if Multichannel is activated (to bypass 3D Secure)', sibsConstants::SIBS_TRANSLATE),
            ];
        }

        return $form_fields;
    }

    /**
     * SIBS Backend Payment Title.
     *
     * @param string $payment_id
     * @return string
     */
    public static function sibs_backend_payment_title($payment_id)
    {
        $paymentName = sibsGeneralFunctions::sibs_translate_backend_payment($payment_id);

        return $paymentName;
    }

    /**
     * SIBS Backend Payment Description.
     *
     * @param string $payment_id
     * @return string
     */
    public static function sibs_backend_payment_desc($payment_id)
    {
        switch($payment_id) {
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_CREDIT_CARD:
                return __('Accept Card payments from VISA and MasterCard', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_BLIK:
                return __('Allow your clients to purchase online by authorizing the purchase in the Blik app', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_PAY_BY_LINK:
                return __('Allow your clients to authorize purchase through Pay By Link', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_PAY_BY_LINK_KEVIN:
                return __('Allow your clients to authorize purchase through Pay By Link Kevin', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_XPAY:
                return __('Allow your clients to purchase through Xpay', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_APAY:
                return __('Allow your clients to purchase through Apple Pay', sibsConstants::SIBS_TRANSLATE);

                break;
            case sibsConstants::SIBS_PAYMENT_METHOD_ORDER_GPAY:
                return __('Allow your clients to purchase through Google Pay', sibsConstants::SIBS_TRANSLATE);

                break;
            default:
                return '';
        }
    }
}
