<?php

namespace Sibs;

use Exception;
use Throwable;
use Context;

class SibsException extends Exception
{
    /**
     * @var string
     */
    protected string $errorIdentifier;

    /**
     * Constructor.
     *
     * @param string         $errorIdentifier
     * @param int            $status
     * @param Throwable|null $previous
     */
    public function __construct(string $errorIdentifier = 'ERROR_UNKNOWN', int $status = 0, ?Throwable $previous = null)
    {
        $this->errorIdentifier = $errorIdentifier;
        $message = $this->getErrorTranslatedMessage($errorIdentifier);

        parent::__construct($message, $status, $previous);

        SibsLogger::error($this->getMessage());
    }

    /**
     * Get Error Message.
     *
     * @return string
     */
    public function getErrorMessage()
    {
        return $this->getErrorTranslatedMessage($this->errorIdentifier);
    }

    /**
     * Returns the translated error message.
     *
     * @param string $errorIdentifier
     * @return string
     */
    protected function getErrorTranslatedMessage(string $errorIdentifier): string
    {
        $translator = Context::getContext()->getTranslator();

        $messages = [
            'ERROR_PARAMETER_CART' => 'Please fill your shopping cart to make payment with Klarna.',
            'ERROR_ORDER_INVALID' => 'Your cart or your information is not complete. Please complete the information before make a payment.',
            'ERROR_MESSAGE_PCLASS_REQUIRED' => 'Transaction cannot be processed because PCLASS parameter is missing, please contact the shop admin for further information.',
            'ERROR_GENERAL_NORESPONSE' => 'Unfortunately, the confirmation of your payment failed. Please contact your merchant for clarification.',
            'ERROR_GENERAL_FRAUD_DETECTION' => 'Unfortunately, there was an error while processing your order. In case a payment has been made, it will be automatically refunded.',
            'ERROR_CC_ACCOUNT' => 'The account holder entered does not match your name. Please use an account that is registered on your name.',
            'ERROR_CC_INVALIDDATA' => 'Unfortunately, the card/account data you entered was not correct. Please try again.',
            'ERROR_CC_BLACKLIST' => 'Unfortunately, the credit card you entered can not be accepted. Please choose a different card or payment method.',
            'ERROR_CC_DECLINED_CARD' => 'Unfortunately, the credit card you entered can not be accepted. Please choose a different card or payment method.',
            'ERROR_CC_EXPIRED' => 'Unfortunately, the credit card you entered is expired. Please choose a different card or payment method.',
            'ERROR_CC_INVALIDCVV' => 'Unfortunately, the CVV/CVC you entered is not correct. Please try again.',
            'ERROR_CC_EXPIRY' => 'Unfortunately, the expiration date you entered is not correct. Please try again.',
            'ERROR_CC_LIMIT_EXCEED' => 'Unfortunately, the limit of your credit card is exceeded. Please choose a different card or payment method.',
            'ERROR_CC_3DAUTH' => 'Unfortunately, the password you entered was not correct. Please try again.',
            'ERROR_CC_3DERROR' => 'Unfortunately, there has been an error while processing your request. Please try again.',
            'ERROR_CC_NOBRAND' => 'Unfortunately, there has been an error while processing your request. Please try again.',
            'ERROR_GENERAL_LIMIT_AMOUNT' => 'Unfortunately, your credit limit is exceeded. Please choose a different card or payment method.',
            'ERROR_GENERAL_LIMIT_TRANSACTIONS' => 'Unfortunately, your limit of transaction is exceeded. Please try again later.',
            'ERROR_CC_DECLINED_AUTH' => 'Unfortunately, your transaction has failed. Please choose a different card or payment method.',
            'ERROR_GENERAL_DECLINED_RISK' => 'Unfortunately, your transaction has failed. Please choose a different card or payment method.',
            'ERROR_CC_ADDRESS' => 'We are sorry. We could not accept your card as its origin does not match your address.',
            'ERROR_GENERAL_CANCEL' => 'You cancelled the payment prior to its execution. Please try again.',
            'ERROR_CC_RECURRING' => 'Recurring transactions have been deactivated for this credit card. Please choose a different card or payment method.',
            'ERROR_CC_REPEATED' => 'Unfortunately, your transaction has been declined due to invalid data. Please choose a different card or payment method.',
            'ERROR_GENERAL_ADDRESS' => 'Unfortunately, your transaction has failed. Please check the personal data you entered.',
            'ERROR_GENERAL_BLACKLIST' => 'The chosen payment method is not available at the moment. Please choose a different card or payment method.',
            'ERROR_GENERAL_GENERAL' => 'Unfortunately, your transaction has failed. Please try again.',
            'ERROR_GENERAL_REDIRECT' => 'Unfortunately, your transaction has failed. Please try again.',
            'ERROR_GENERAL_TIMEOUT' => 'Unfortunately, your transaction has failed. Please try again.',
            'ERROR_GIRO_NOSUPPORT' => 'Giropay is not supported for this transaction. Please choose a different payment method.',
            'ERROR_ADDRESS_PHONE' => 'Unfortunately, your transaction has failed. Please enter a valid telephone number.',
            'ERROR_UNKNOWN' => 'Unfortunately, your transaction has failed. Please try again.',
            'ERROR_MC_UPDATE' => 'We are sorry. Your attempt to update your payment information was not successful.',
            'ERROR_MC_ADD' => 'We are sorry. Your attempt to save your payment information was not successful.',
            'ERROR_MC_DELETE' => 'We are sorry. Your attempt to delete your payment information was not successful.',
            'ERROR_CHARS' => 'We are sorry. Error on chars.',
        ];

        $message = $messages[$errorIdentifier] ?? $messages['ERROR_UNKNOWN'];

        return $translator->trans(
            $message,
            [],
            'Modules.Sibs.Errors'
        );
    }
}
